// Copyright 2014 The Rust Project Developers. See the COPYRIGHT
// file at the top-level directory of this distribution and at
// http://rust-lang.org/COPYRIGHT.
//
// Licensed under the Apache License, Version 2.0 <LICENSE-APACHE or
// http://www.apache.org/licenses/LICENSE-2.0> or the MIT license
// <LICENSE-MIT or http://opensource.org/licenses/MIT>, at your
// option. This file may not be copied, modified, or distributed
// except according to those terms.

//! A typesafe bitmask flag generator useful for sets of C-style flags.
//! It can be used for creating ergonomic wrappers around C APIs.
//!
//! The `bitflags!` macro generates `struct`s that manage a set of flags. The
//! type of those flags must be some primitive integer.
//!
//! # Examples
//!
//! ```
//! use bitflags::bitflags;
//!
//! bitflags! {
//!     #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
//!     struct Flags: u32 {
//!         const A = 0b00000001;
//!         const B = 0b00000010;
//!         const C = 0b00000100;
//!         const ABC = Self::A.bits() | Self::B.bits() | Self::C.bits();
//!     }
//! }
//!
//! fn main() {
//!     let e1 = Flags::A | Flags::C;
//!     let e2 = Flags::B | Flags::C;
//!     assert_eq!((e1 | e2), Flags::ABC);   // union
//!     assert_eq!((e1 & e2), Flags::C);     // intersection
//!     assert_eq!((e1 - e2), Flags::A);     // set difference
//!     assert_eq!(!e2, Flags::A);           // set complement
//! }
//! ```
//!
//! See [`example_generated::Flags`](./example_generated/struct.Flags.html) for documentation of code
//! generated by the above `bitflags!` expansion.
//!
//! # Visibility
//!
//! The `bitflags!` macro supports visibility, just like you'd expect when writing a normal
//! Rust `struct`:
//!
//! ```
//! mod example {
//!     use bitflags::bitflags;
//!
//!     bitflags! {
//!         #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
//!         pub struct Flags1: u32 {
//!             const A = 0b00000001;
//!         }
//!
//!         #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
//! #       pub
//!         struct Flags2: u32 {
//!             const B = 0b00000010;
//!         }
//!     }
//! }
//!
//! fn main() {
//!     let flag1 = example::Flags1::A;
//!     let flag2 = example::Flags2::B; // error: const `B` is private
//! }
//! ```
//!
//! # Attributes
//!
//! Attributes can be attached to the generated flags types and their constants as normal.
//!
//! # Representation
//!
//! It's valid to add a `#[repr(C)]` or `#[repr(transparent)]` attribute to a generated flags type.
//! The generated flags type is always guaranteed to be a newtype where its only field has the same
//! ABI as the underlying integer type.
//!
//! In this example, `Flags` has the same ABI as `u32`:
//!
//! ```
//! use bitflags::bitflags;
//!
//! bitflags! {
//!     #[repr(transparent)]
//!     #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
//!     struct Flags: u32 {
//!         const A = 0b00000001;
//!         const B = 0b00000010;
//!         const C = 0b00000100;
//!     }
//! }
//! ```
//!
//! # Extending
//!
//! Generated flags types belong to you, so you can add trait implementations to them outside
//! of what the `bitflags!` macro gives:
//!
//! ```
//! use std::fmt;
//!
//! use bitflags::bitflags;
//!
//! bitflags! {
//!     #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
//!     struct Flags: u32 {
//!         const A = 0b00000001;
//!         const B = 0b00000010;
//!     }
//! }
//!
//! impl Flags {
//!     pub fn clear(&mut self) {
//!         *self.0.bits_mut() = 0;
//!     }
//! }
//!
//! fn main() {
//!     let mut flags = Flags::A | Flags::B;
//!
//!     flags.clear();
//!     assert!(flags.is_empty());
//!
//!     assert_eq!(format!("{:?}", Flags::A | Flags::B), "Flags(A | B)");
//!     assert_eq!(format!("{:?}", Flags::B), "Flags(B)");
//! }
//! ```
//!
//! # What's implemented by `bitflags!`
//!
//! The `bitflags!` macro adds some trait implementations and inherent methods
//! to generated flags types, but leaves room for you to choose the semantics
//! of others.
//!
//! ## Iterators
//!
//! The following iterator traits are implemented for generated flags types:
//!
//! - `Extend`: adds the union of the instances iterated over.
//! - `FromIterator`: calculates the union.
//! - `IntoIterator`: iterates over set flag values.
//!
//! ## Formatting
//!
//! The following formatting traits are implemented for generated flags types:
//!
//! - `Binary`.
//! - `LowerHex` and `UpperHex`.
//! - `Octal`.
//!
//! Also see the _Debug and Display_ section for details about standard text
//! representations for flags types.
//!
//! ## Operators
//!
//! The following operator traits are implemented for the generated `struct`s:
//!
//! - `BitOr` and `BitOrAssign`: union
//! - `BitAnd` and `BitAndAssign`: intersection
//! - `BitXor` and `BitXorAssign`: toggle
//! - `Sub` and `SubAssign`: set difference
//! - `Not`: set complement
//!
//! ## Methods
//!
//! The following methods are defined for the generated `struct`s:
//!
//! - `empty`: an empty set of flags
//! - `all`: the set of all defined flags
//! - `bits`: the raw value of the flags currently stored
//! - `from_bits`: convert from underlying bit representation, unless that
//!                representation contains bits that do not correspond to a
//!                defined flag
//! - `from_bits_truncate`: convert from underlying bit representation, dropping
//!                         any bits that do not correspond to defined flags
//! - `from_bits_retain`: convert from underlying bit representation, keeping
//!                          all bits (even those not corresponding to defined
//!                          flags)
//! - `is_empty`: `true` if no flags are currently stored
//! - `is_all`: `true` if currently set flags exactly equal all defined flags
//! - `intersects`: `true` if there are flags common to both `self` and `other`
//! - `contains`: `true` if all of the flags in `other` are contained within `self`
//! - `insert`: inserts the specified flags in-place
//! - `remove`: removes the specified flags in-place
//! - `toggle`: the specified flags will be inserted if not present, and removed
//!             if they are.
//! - `set`: inserts or removes the specified flags depending on the passed value
//! - `intersection`: returns a new set of flags, containing only the flags present
//!                   in both `self` and `other` (the argument to the function).
//! - `union`: returns a new set of flags, containing any flags present in
//!            either `self` or `other` (the argument to the function).
//! - `difference`: returns a new set of flags, containing all flags present in
//!                 `self` without any of the flags present in `other` (the
//!                 argument to the function).
//! - `symmetric_difference`: returns a new set of flags, containing all flags
//!                           present in either `self` or `other` (the argument
//!                           to the function), but not both.
//! - `complement`: returns a new set of flags, containing all flags which are
//!                 not set in `self`, but which are allowed for this type.
//!
//! # What's not implemented by `bitflags!`
//!
//! Some functionality is not automatically implemented for generated flags types
//! by the `bitflags!` macro, even when it reasonably could be. This is so callers
//! have more freedom to decide on the semantics of their flags types.
//!
//! ## `Clone` and `Copy`
//!
//! Generated flags types are not automatically copyable, even though they can always
//! derive both `Clone` and `Copy`.
//!
//! ## `Default`
//!
//! The `Default` trait is not automatically implemented for the generated structs.
//!
//! If your default value is equal to `0` (which is the same value as calling `empty()`
//! on the generated struct), you can simply derive `Default`:
//!
//! ```
//! use bitflags::bitflags;
//!
//! bitflags! {
//!     // Results in default value with bits: 0
//!     #[derive(Default, Clone, Copy, Debug, PartialEq, Eq, Hash)]
//!     struct Flags: u32 {
//!         const A = 0b00000001;
//!         const B = 0b00000010;
//!         const C = 0b00000100;
//!     }
//! }
//!
//! fn main() {
//!     let derived_default: Flags = Default::default();
//!     assert_eq!(derived_default.bits(), 0);
//! }
//! ```
//!
//! If your default value is not equal to `0` you need to implement `Default` yourself:
//!
//! ```
//! use bitflags::bitflags;
//!
//! bitflags! {
//!     #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
//!     struct Flags: u32 {
//!         const A = 0b00000001;
//!         const B = 0b00000010;
//!         const C = 0b00000100;
//!     }
//! }
//!
//! // explicit `Default` implementation
//! impl Default for Flags {
//!     fn default() -> Flags {
//!         Flags::A | Flags::C
//!     }
//! }
//!
//! fn main() {
//!     let implemented_default: Flags = Default::default();
//!     assert_eq!(implemented_default, (Flags::A | Flags::C));
//! }
//! ```
//!
//! ## `Debug` and `Display`
//!
//! The `Debug` trait can be derived for a reasonable implementation. This library defines a standard
//! text-based representation for flags that generated flags types can use. For details on the exact
//! grammar, see the [`parser`] module.
//!
//! To support formatting and parsing your generated flags types using that representation, you can implement
//! the standard `Display` and `FromStr` traits in this fashion:
//!
//! ```
//! use bitflags::bitflags;
//! use std::{fmt, str};
//!
//! bitflags::bitflags! {
//!     pub struct Flags: u32 {
//!         const A = 1;
//!         const B = 2;
//!         const C = 4;
//!         const D = 8;
//!     }
//! }
//!
//! impl fmt::Debug for Flags {
//!     fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
//!         fmt::Debug::fmt(&self.0, f)
//!     }
//! }
//!
//! impl fmt::Display for Flags {
//!     fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
//!         fmt::Display::fmt(&self.0, f)
//!     }
//! }
//!
//! impl str::FromStr for Flags {
//!     type Err = bitflags::parser::ParseError;
//!
//!     fn from_str(flags: &str) -> Result<Self, Self::Err> {
//!         Ok(Self(flags.parse()?))
//!     }
//! }
//! ```
//!
//! ## `PartialEq` and `PartialOrd`
//!
//! Equality and ordering can be derived for a reasonable implementation, or implemented manually
//! for different semantics.
//!
//! # Edge cases
//!
//! ## Zero Flags
//!
//! Flags with a value equal to zero will have some strange behavior that one should be aware of.
//!
//! ```
//! use bitflags::bitflags;
//!
//! bitflags! {
//!     #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
//!     struct Flags: u32 {
//!         const NONE = 0b00000000;
//!         const SOME = 0b00000001;
//!     }
//! }
//!
//! fn main() {
//!     let empty = Flags::empty();
//!     let none = Flags::NONE;
//!     let some = Flags::SOME;
//!
//!     // Zero flags are treated as always present
//!     assert!(empty.contains(Flags::NONE));
//!     assert!(none.contains(Flags::NONE));
//!     assert!(some.contains(Flags::NONE));
//!
//!     // Zero flags will be ignored when testing for emptiness
//!     assert!(none.is_empty());
//! }
//! ```
//!
//! Users should generally avoid defining a flag with a value of zero.
//!
//! ## Multi-bit Flags
//!
//! It is allowed to define a flag with multiple bits set, however such
//! flags are _not_ treated as a set where any of those bits is a valid
//! flag. Instead, each flag is treated as a unit when converting from
//! bits with [`from_bits`] or [`from_bits_truncate`].
//!
//! ```
//! use bitflags::bitflags;
//!
//! bitflags! {
//!     #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
//!     struct Flags: u8 {
//!         const F3 = 0b00000011;
//!     }
//! }
//!
//! fn main() {
//!     // This bit pattern does not set all the bits in `F3`, so it is rejected.
//!     assert!(Flags::from_bits(0b00000001).is_none());
//!     assert!(Flags::from_bits_truncate(0b00000001).is_empty());
//! }
//! ```
//!
//! [`from_bits`]: BitFlags::from_bits
//! [`from_bits_truncate`]: BitFlags::from_bits_truncate
//!
//! # The `BitFlags` trait
//!
//! This library defines a `BitFlags` trait that's implemented by all generated flags types.
//! The trait makes it possible to work with flags types generically:
//!
//! ```
//! fn count_unset_flags<F: bitflags::BitFlags>(flags: &F) -> usize {
//!     // Find out how many flags there are in total
//!     let total = F::all().iter().count();
//!
//!     // Find out how many flags are set
//!     let set = flags.iter().count();
//!
//!     total - set
//! }
//!
//! use bitflags::bitflags;
//!
//! bitflags! {
//!     #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
//!     struct Flags: u32 {
//!         const A = 0b00000001;
//!         const B = 0b00000010;
//!         const C = 0b00000100;
//!     }
//! }
//!
//! assert_eq!(2, count_unset_flags(&Flags::B));
//! ```
//!
//! # The internal field
//!
//! This library generates newtypes like:
//!
//! ```
//! # pub struct Field0;
//! pub struct Flags(Field0);
//! ```
//!
//! You can freely use methods and trait implementations on this internal field as `.0`.
//! For details on exactly what's generated for it, see the [`Field0`](example_generated/struct.Field0.html)
//! example docs.

#![cfg_attr(not(any(feature = "std", test)), no_std)]
#![cfg_attr(not(test), forbid(unsafe_code))]

#![doc(html_root_url = "https://docs.rs/bitflags/2.2.1")]

#[doc(inline)]
pub use traits::BitFlags;

pub mod parser;
mod traits;

#[doc(hidden)]
pub mod __private {
    pub use crate::{external::*, traits::*};

    pub use core;
}

/*
How does the bitflags crate work?

This library generates a `struct` in the end-user's crate with a bunch of constants on it that represent flags.
The difference between `bitflags` and a lot of other libraries is that we don't actually control the generated `struct` in the end.
It's part of the end-user's crate, so it belongs to them. That makes it difficult to extend `bitflags` with new functionality
because we could end up breaking valid code that was already written.

Our solution is to split the type we generate into two: the public struct owned by the end-user, and an internal struct owned by `bitflags` (us).
To give you an example, let's say we had a crate that called `bitflags!`:

```rust
bitflags! {
    pub struct MyFlags: u32 {
        const A = 1;
        const B = 2;
    }
}
```

What they'd end up with looks something like this:

```rust
pub struct MyFlags(<MyFlags as PublicFlags>::InternalBitFlags);

const _: () = {
    #[repr(transparent)]
    #[derive(Clone, Copy, PartialEq, Eq, PartialOrd, Ord, Hash)]
    pub struct MyInternalBitFlags {
        bits: u32,
    }

    impl PublicFlags for MyFlags {
        type Internal = InternalBitFlags;
    }
};
```

If we want to expose something like a new trait impl for generated flags types, we add it to our generated `MyInternalBitFlags`,
and let `#[derive]` on `MyFlags` pick up that implementation, if an end-user chooses to add one.

The public API is generated in the `__impl_public_flags!` macro, and the internal API is generated in
the `__impl_internal_flags!` macro.

The macros are split into 3 modules:

- `public`: where the user-facing flags types are generated.
- `internal`: where the `bitflags`-facing flags types are generated.
- `external`: where external library traits are implemented conditionally.
*/

/// The macro used to generate the flag structure.
///
/// See the [crate level docs](../bitflags/index.html) for complete documentation.
///
/// # Example
///
/// ```
/// use bitflags::bitflags;
///
/// bitflags! {
///     #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
///     struct Flags: u32 {
///         const A = 0b00000001;
///         const B = 0b00000010;
///         const C = 0b00000100;
///         const ABC = Self::A.bits() | Self::B.bits() | Self::C.bits();
///     }
/// }
///
/// let e1 = Flags::A | Flags::C;
/// let e2 = Flags::B | Flags::C;
/// assert_eq!((e1 | e2), Flags::ABC);   // union
/// assert_eq!((e1 & e2), Flags::C);     // intersection
/// assert_eq!((e1 - e2), Flags::A);     // set difference
/// assert_eq!(!e2, Flags::A);           // set complement
/// ```
///
/// The generated `struct`s can also be extended with type and trait
/// implementations:
///
/// ```
/// use std::fmt;
///
/// use bitflags::bitflags;
///
/// bitflags! {
///     #[derive(Clone, Copy, Debug, PartialEq, Eq, Hash)]
///     struct Flags: u32 {
///         const A = 0b00000001;
///         const B = 0b00000010;
///     }
/// }
///
/// impl Flags {
///     pub fn clear(&mut self) {
///         *self.0.bits_mut() = 0;
///     }
/// }
///
/// let mut flags = Flags::A | Flags::B;
///
/// flags.clear();
/// assert!(flags.is_empty());
///
/// assert_eq!(format!("{:?}", Flags::A | Flags::B), "Flags(A | B)");
/// assert_eq!(format!("{:?}", Flags::B), "Flags(B)");
/// ```
#[macro_export(local_inner_macros)]
macro_rules! bitflags {
    (
        $(#[$outer:meta])*
        $vis:vis struct $BitFlags:ident: $T:ty {
            $(
                $(#[$inner:ident $($args:tt)*])*
                const $Flag:ident = $value:expr;
            )*
        }

        $($t:tt)*
    ) => {
        // Declared in the scope of the `bitflags!` call
        // This type appears in the end-user's API
        __declare_public_bitflags! {
            $(#[$outer])*
            $vis struct $BitFlags;
        }

        // Workaround for: https://github.com/bitflags/bitflags/issues/320
        __impl_public_bitflags_consts! {
            $BitFlags {
                $(
                    $(#[$inner $($args)*])*
                    #[allow(
                        dead_code,
                        deprecated,
                        unused_attributes,
                        non_upper_case_globals
                    )]
                    $Flag = $value;
                )*
            }
        }

        #[allow(
            dead_code,
            deprecated,
            unused_doc_comments,
            unused_attributes,
            unused_mut,
            unused_imports,
            non_upper_case_globals
        )]
        const _: () = {
            // Declared in a "hidden" scope that can't be reached directly
            // These types don't appear in the end-user's API
            __declare_internal_bitflags! {
                $vis struct InternalBitFlags: $T;
                $vis struct Iter;
                $vis struct IterRaw;
            }

            __impl_internal_bitflags! {
                InternalBitFlags: $T, $BitFlags, Iter, IterRaw {
                    $(
                        $(#[$inner $($args)*])*
                        $Flag;
                    )*
                }
            }

            // This is where new library trait implementations can be added
            __impl_external_bitflags! {
                InternalBitFlags: $T {
                    $(
                        $(#[$inner $($args)*])*
                        $Flag;
                    )*
                }
            }

            __impl_public_bitflags! {
                $BitFlags: $T, InternalBitFlags, Iter, IterRaw;
            }
        };

        bitflags! {
            $($t)*
        }
    };
    () => {};
}

#[macro_use]
mod public;
#[macro_use]
mod internal;
#[macro_use]
mod external;

#[cfg(feature = "example_generated")]
pub mod example_generated;

#[cfg(test)]
mod tests {
    use std::{
        collections::hash_map::DefaultHasher,
        fmt,
        hash::{Hash, Hasher},
        str,
    };

    bitflags! {
        #[doc = "> The first principle is that you must not fool yourself — and"]
        #[doc = "> you are the easiest person to fool."]
        #[doc = "> "]
        #[doc = "> - Richard Feynman"]
        #[derive(Clone, Copy, Default, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
        struct Flags: u32 {
            const A = 0b00000001;
            #[doc = "<pcwalton> macros are way better at generating code than trans is"]
            const B = 0b00000010;
            const C = 0b00000100;
            #[doc = "* cmr bed"]
            #[doc = "* strcat table"]
            #[doc = "<strcat> wait what?"]
            const ABC = Self::A.bits() | Self::B.bits() | Self::C.bits();
        }

        #[derive(Clone, Copy, Default, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
        struct _CfgFlags: u32 {
            #[cfg(unix)]
            const _CFG_A = 0b01;
            #[cfg(windows)]
            const _CFG_B = 0b01;
            #[cfg(unix)]
            const _CFG_C = Self::_CFG_A.bits() | 0b10;
        }

        #[derive(Clone, Copy, Default, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
        struct AnotherSetOfFlags: i8 {
            const ANOTHER_FLAG = -1_i8;
        }

        #[derive(Clone, Copy, Default, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
        struct LongFlags: u32 {
            const LONG_A = 0b1111111111111111;
        }
    }

    bitflags! {
        #[derive(Debug, PartialEq, Eq)]
        struct FmtFlags: u16 {
            const 고양이 = 0b0000_0001;
            const 개 = 0b0000_0010;
            const 물고기 = 0b0000_0100;
            const 물고기_고양이 = Self::고양이.bits() | Self::물고기.bits();
        }
    }

    impl str::FromStr for FmtFlags {
        type Err = crate::parser::ParseError;

        fn from_str(flags: &str) -> Result<Self, Self::Err> {
            Ok(Self(flags.parse()?))
        }
    }

    impl fmt::Display for FmtFlags {
        fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
            fmt::Display::fmt(&self.0, f)
        }
    }

    bitflags! {
        #[derive(Clone, Copy, Default, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
        struct EmptyFlags: u32 {
        }
    }

    #[test]
    fn test_bits() {
        assert_eq!(Flags::empty().bits(), 0b00000000);
        assert_eq!(Flags::A.bits(), 0b00000001);
        assert_eq!(Flags::ABC.bits(), 0b00000111);

        assert_eq!(AnotherSetOfFlags::empty().bits(), 0b00);
        assert_eq!(AnotherSetOfFlags::ANOTHER_FLAG.bits(), !0_i8);

        assert_eq!(EmptyFlags::empty().bits(), 0b00000000);
    }

    #[test]
    fn test_from_bits() {
        assert_eq!(Flags::from_bits(0), Some(Flags::empty()));
        assert_eq!(Flags::from_bits(0b1), Some(Flags::A));
        assert_eq!(Flags::from_bits(0b10), Some(Flags::B));
        assert_eq!(Flags::from_bits(0b11), Some(Flags::A | Flags::B));
        assert_eq!(Flags::from_bits(0b1000), None);

        assert_eq!(
            AnotherSetOfFlags::from_bits(!0_i8),
            Some(AnotherSetOfFlags::ANOTHER_FLAG)
        );

        assert_eq!(EmptyFlags::from_bits(0), Some(EmptyFlags::empty()));
        assert_eq!(EmptyFlags::from_bits(0b1), None);
    }

    #[test]
    fn test_from_bits_truncate() {
        assert_eq!(Flags::from_bits_truncate(0), Flags::empty());
        assert_eq!(Flags::from_bits_truncate(0b1), Flags::A);
        assert_eq!(Flags::from_bits_truncate(0b10), Flags::B);
        assert_eq!(Flags::from_bits_truncate(0b11), (Flags::A | Flags::B));
        assert_eq!(Flags::from_bits_truncate(0b1000), Flags::empty());
        assert_eq!(Flags::from_bits_truncate(0b1001), Flags::A);

        assert_eq!(
            AnotherSetOfFlags::from_bits_truncate(0_i8),
            AnotherSetOfFlags::empty()
        );

        assert_eq!(EmptyFlags::from_bits_truncate(0), EmptyFlags::empty());
        assert_eq!(EmptyFlags::from_bits_truncate(0b1), EmptyFlags::empty());
    }

    #[test]
    fn test_from_bits_retain() {
        let extra = Flags::from_bits_retain(0b1000);
        assert_eq!(Flags::from_bits_retain(0), Flags::empty());
        assert_eq!(Flags::from_bits_retain(0b1), Flags::A);
        assert_eq!(Flags::from_bits_retain(0b10), Flags::B);

        assert_eq!(Flags::from_bits_retain(0b11), (Flags::A | Flags::B));
        assert_eq!(Flags::from_bits_retain(0b1000), (extra | Flags::empty()));
        assert_eq!(Flags::from_bits_retain(0b1001), (extra | Flags::A));

        let extra = EmptyFlags::from_bits_retain(0b1000);
        assert_eq!(
            EmptyFlags::from_bits_retain(0b1000),
            (extra | EmptyFlags::empty())
        );
    }

    #[test]
    fn test_is_empty() {
        assert!(Flags::empty().is_empty());
        assert!(!Flags::A.is_empty());
        assert!(!Flags::ABC.is_empty());

        assert!(!AnotherSetOfFlags::ANOTHER_FLAG.is_empty());

        assert!(EmptyFlags::empty().is_empty());
        assert!(EmptyFlags::all().is_empty());
    }

    #[test]
    fn test_is_all() {
        assert!(Flags::all().is_all());
        assert!(!Flags::A.is_all());
        assert!(Flags::ABC.is_all());

        let extra = Flags::from_bits_retain(0b1000);
        assert!(!extra.is_all());
        assert!(!(Flags::A | extra).is_all());
        assert!((Flags::ABC | extra).is_all());

        assert!(AnotherSetOfFlags::ANOTHER_FLAG.is_all());

        assert!(EmptyFlags::all().is_all());
        assert!(EmptyFlags::empty().is_all());
    }

    #[test]
    fn test_two_empties_do_not_intersect() {
        let e1 = Flags::empty();
        let e2 = Flags::empty();
        assert!(!e1.intersects(e2));

        assert!(AnotherSetOfFlags::ANOTHER_FLAG.intersects(AnotherSetOfFlags::ANOTHER_FLAG));
    }

    #[test]
    fn test_empty_does_not_intersect_with_full() {
        let e1 = Flags::empty();
        let e2 = Flags::ABC;
        assert!(!e1.intersects(e2));
    }

    #[test]
    fn test_disjoint_intersects() {
        let e1 = Flags::A;
        let e2 = Flags::B;
        assert!(!e1.intersects(e2));
    }

    #[test]
    fn test_overlapping_intersects() {
        let e1 = Flags::A;
        let e2 = Flags::A | Flags::B;
        assert!(e1.intersects(e2));
    }

    #[test]
    fn test_contains() {
        let e1 = Flags::A;
        let e2 = Flags::A | Flags::B;
        assert!(!e1.contains(e2));
        assert!(e2.contains(e1));
        assert!(Flags::ABC.contains(e2));

        assert!(AnotherSetOfFlags::ANOTHER_FLAG.contains(AnotherSetOfFlags::ANOTHER_FLAG));

        assert!(EmptyFlags::empty().contains(EmptyFlags::empty()));
    }

    #[test]
    fn test_insert() {
        let mut e1 = Flags::A;
        let e2 = Flags::A | Flags::B;
        e1.insert(e2);
        assert_eq!(e1, e2);

        let mut e3 = AnotherSetOfFlags::empty();
        e3.insert(AnotherSetOfFlags::ANOTHER_FLAG);
        assert_eq!(e3, AnotherSetOfFlags::ANOTHER_FLAG);
    }

    #[test]
    fn test_remove() {
        let mut e1 = Flags::A | Flags::B;
        let e2 = Flags::A | Flags::C;
        e1.remove(e2);
        assert_eq!(e1, Flags::B);

        let mut e3 = AnotherSetOfFlags::ANOTHER_FLAG;
        e3.remove(AnotherSetOfFlags::ANOTHER_FLAG);
        assert_eq!(e3, AnotherSetOfFlags::empty());
    }

    #[test]
    fn test_operators() {
        let e1 = Flags::A | Flags::C;
        let e2 = Flags::B | Flags::C;
        assert_eq!((e1 | e2), Flags::ABC); // union
        assert_eq!((e1 & e2), Flags::C); // intersection
        assert_eq!((e1 - e2), Flags::A); // set difference
        assert_eq!(!e2, Flags::A); // set complement
        assert_eq!(e1 ^ e2, Flags::A | Flags::B); // toggle
        let mut e3 = e1;
        e3.toggle(e2);
        assert_eq!(e3, Flags::A | Flags::B);

        let mut m4 = AnotherSetOfFlags::empty();
        m4.toggle(AnotherSetOfFlags::empty());
        assert_eq!(m4, AnotherSetOfFlags::empty());
    }

    #[test]
    fn test_operators_unchecked() {
        let extra = Flags::from_bits_retain(0b1000);
        let e1 = Flags::A | Flags::C | extra;
        let e2 = Flags::B | Flags::C;
        assert_eq!((e1 | e2), (Flags::ABC | extra)); // union
        assert_eq!((e1 & e2), Flags::C); // intersection
        assert_eq!((e1 - e2), (Flags::A | extra)); // set difference
        assert_eq!(!e2, Flags::A); // set complement
        assert_eq!(!e1, Flags::B); // set complement
        assert_eq!(e1 ^ e2, Flags::A | Flags::B | extra); // toggle
        let mut e3 = e1;
        e3.toggle(e2);
        assert_eq!(e3, Flags::A | Flags::B | extra);
    }

    #[test]
    fn test_set_ops_basic() {
        let ab = Flags::A.union(Flags::B);
        let ac = Flags::A.union(Flags::C);
        let bc = Flags::B.union(Flags::C);
        assert_eq!(ab.bits(), 0b011);
        assert_eq!(bc.bits(), 0b110);
        assert_eq!(ac.bits(), 0b101);

        assert_eq!(ab, Flags::B.union(Flags::A));
        assert_eq!(ac, Flags::C.union(Flags::A));
        assert_eq!(bc, Flags::C.union(Flags::B));

        assert_eq!(ac, Flags::A | Flags::C);
        assert_eq!(bc, Flags::B | Flags::C);
        assert_eq!(ab.union(bc), Flags::ABC);

        assert_eq!(ac, Flags::A | Flags::C);
        assert_eq!(bc, Flags::B | Flags::C);

        assert_eq!(ac.union(bc), ac | bc);
        assert_eq!(ac.union(bc), Flags::ABC);
        assert_eq!(bc.union(ac), Flags::ABC);

        assert_eq!(ac.intersection(bc), ac & bc);
        assert_eq!(ac.intersection(bc), Flags::C);
        assert_eq!(bc.intersection(ac), Flags::C);

        assert_eq!(ac.difference(bc), ac - bc);
        assert_eq!(bc.difference(ac), bc - ac);
        assert_eq!(ac.difference(bc), Flags::A);
        assert_eq!(bc.difference(ac), Flags::B);

        assert_eq!(bc.complement(), !bc);
        assert_eq!(bc.complement(), Flags::A);
        assert_eq!(ac.symmetric_difference(bc), Flags::A.union(Flags::B));
        assert_eq!(bc.symmetric_difference(ac), Flags::A.union(Flags::B));
    }

    #[test]
    fn test_set_ops_const() {
        // These just test that these compile and don't cause use-site panics
        // (would be possible if we had some sort of UB)
        const INTERSECT: Flags = Flags::all().intersection(Flags::C);
        const UNION: Flags = Flags::A.union(Flags::C);
        const DIFFERENCE: Flags = Flags::all().difference(Flags::A);
        const COMPLEMENT: Flags = Flags::C.complement();
        const SYM_DIFFERENCE: Flags = UNION.symmetric_difference(DIFFERENCE);
        assert_eq!(INTERSECT, Flags::C);
        assert_eq!(UNION, Flags::A | Flags::C);
        assert_eq!(DIFFERENCE, Flags::all() - Flags::A);
        assert_eq!(COMPLEMENT, !Flags::C);
        assert_eq!(
            SYM_DIFFERENCE,
            (Flags::A | Flags::C) ^ (Flags::all() - Flags::A)
        );
    }

    #[test]
    fn test_set_ops_unchecked() {
        let extra = Flags::from_bits_retain(0b1000);
        let e1 = Flags::A.union(Flags::C).union(extra);
        let e2 = Flags::B.union(Flags::C);
        assert_eq!(e1.bits(), 0b1101);
        assert_eq!(e1.union(e2), (Flags::ABC | extra));
        assert_eq!(e1.intersection(e2), Flags::C);
        assert_eq!(e1.difference(e2), Flags::A | extra);
        assert_eq!(e2.difference(e1), Flags::B);
        assert_eq!(e2.complement(), Flags::A);
        assert_eq!(e1.complement(), Flags::B);
        assert_eq!(e1.symmetric_difference(e2), Flags::A | Flags::B | extra); // toggle
    }

    #[test]
    fn test_set_ops_exhaustive() {
        // Define a flag that contains gaps to help exercise edge-cases,
        // especially around "unknown" flags (e.g. ones outside of `all()`
        // `from_bits_retain`).
        // - when lhs and rhs both have different sets of unknown flags.
        // - unknown flags at both ends, and in the middle
        // - cases with "gaps".
        bitflags! {
            #[derive(Clone, Copy, Debug, PartialEq, Eq)]
            struct Test: u16 {
                // Intentionally no `A`
                const B = 0b000000010;
                // Intentionally no `C`
                const D = 0b000001000;
                const E = 0b000010000;
                const F = 0b000100000;
                const G = 0b001000000;
                // Intentionally no `H`
                const I = 0b100000000;
            }
        }
        let iter_test_flags = || (0..=0b111_1111_1111).map(|bits| Test::from_bits_retain(bits));

        for a in iter_test_flags() {
            assert_eq!(
                a.complement(),
                Test::from_bits_truncate(!a.bits()),
                "wrong result: !({:?})",
                a,
            );
            assert_eq!(a.complement(), !a, "named != op: !({:?})", a);
            for b in iter_test_flags() {
                // Check that the named operations produce the expected bitwise
                // values.
                assert_eq!(
                    a.union(b).bits(),
                    a.bits() | b.bits(),
                    "wrong result: `{:?}` | `{:?}`",
                    a,
                    b,
                );
                assert_eq!(
                    a.intersection(b).bits(),
                    a.bits() & b.bits(),
                    "wrong result: `{:?}` & `{:?}`",
                    a,
                    b,
                );
                assert_eq!(
                    a.symmetric_difference(b).bits(),
                    a.bits() ^ b.bits(),
                    "wrong result: `{:?}` ^ `{:?}`",
                    a,
                    b,
                );
                assert_eq!(
                    a.difference(b).bits(),
                    a.bits() & !b.bits(),
                    "wrong result: `{:?}` - `{:?}`",
                    a,
                    b,
                );
                // Note: Difference is checked as both `a - b` and `b - a`
                assert_eq!(
                    b.difference(a).bits(),
                    b.bits() & !a.bits(),
                    "wrong result: `{:?}` - `{:?}`",
                    b,
                    a,
                );
                // Check that the named set operations are equivalent to the
                // bitwise equivalents
                assert_eq!(a.union(b), a | b, "named != op: `{:?}` | `{:?}`", a, b,);
                assert_eq!(
                    a.intersection(b),
                    a & b,
                    "named != op: `{:?}` & `{:?}`",
                    a,
                    b,
                );
                assert_eq!(
                    a.symmetric_difference(b),
                    a ^ b,
                    "named != op: `{:?}` ^ `{:?}`",
                    a,
                    b,
                );
                assert_eq!(a.difference(b), a - b, "named != op: `{:?}` - `{:?}`", a, b,);
                // Note: Difference is checked as both `a - b` and `b - a`
                assert_eq!(b.difference(a), b - a, "named != op: `{:?}` - `{:?}`", b, a,);
                // Verify that the operations which should be symmetric are
                // actually symmetric.
                assert_eq!(a.union(b), b.union(a), "asymmetry: `{:?}` | `{:?}`", a, b,);
                assert_eq!(
                    a.intersection(b),
                    b.intersection(a),
                    "asymmetry: `{:?}` & `{:?}`",
                    a,
                    b,
                );
                assert_eq!(
                    a.symmetric_difference(b),
                    b.symmetric_difference(a),
                    "asymmetry: `{:?}` ^ `{:?}`",
                    a,
                    b,
                );
            }
        }
    }

    #[test]
    fn test_set() {
        let mut e1 = Flags::A | Flags::C;
        e1.set(Flags::B, true);
        e1.set(Flags::C, false);

        assert_eq!(e1, Flags::A | Flags::B);
    }

    #[test]
    fn test_assignment_operators() {
        let mut m1 = Flags::empty();
        let e1 = Flags::A | Flags::C;
        // union
        m1 |= Flags::A;
        assert_eq!(m1, Flags::A);
        // intersection
        m1 &= e1;
        assert_eq!(m1, Flags::A);
        // set difference
        m1 -= m1;
        assert_eq!(m1, Flags::empty());
        // toggle
        m1 ^= e1;
        assert_eq!(m1, e1);
    }

    #[test]
    fn test_const_fn() {
        const _M1: Flags = Flags::empty();

        const M2: Flags = Flags::A;
        assert_eq!(M2, Flags::A);

        const M3: Flags = Flags::C;
        assert_eq!(M3, Flags::C);
    }

    #[test]
    fn test_extend() {
        let mut flags;

        flags = Flags::empty();
        flags.extend([].iter().cloned());
        assert_eq!(flags, Flags::empty());

        flags = Flags::empty();
        flags.extend([Flags::A, Flags::B].iter().cloned());
        assert_eq!(flags, Flags::A | Flags::B);

        flags = Flags::A;
        flags.extend([Flags::A, Flags::B].iter().cloned());
        assert_eq!(flags, Flags::A | Flags::B);

        flags = Flags::B;
        flags.extend([Flags::A, Flags::ABC].iter().cloned());
        assert_eq!(flags, Flags::ABC);
    }

    #[test]
    fn test_from_iterator() {
        assert_eq!([].iter().cloned().collect::<Flags>(), Flags::empty());
        assert_eq!(
            [Flags::A, Flags::B].iter().cloned().collect::<Flags>(),
            Flags::A | Flags::B
        );
        assert_eq!(
            [Flags::A, Flags::ABC].iter().cloned().collect::<Flags>(),
            Flags::ABC
        );
    }

    #[test]
    fn test_lt() {
        let mut a = Flags::empty();
        let mut b = Flags::empty();

        assert!(!(a < b) && !(b < a));
        b = Flags::B;
        assert!(a < b);
        a = Flags::C;
        assert!(!(a < b) && b < a);
        b = Flags::C | Flags::B;
        assert!(a < b);
    }

    #[test]
    fn test_ord() {
        let mut a = Flags::empty();
        let mut b = Flags::empty();

        assert!(a <= b && a >= b);
        a = Flags::A;
        assert!(a > b && a >= b);
        assert!(b < a && b <= a);
        b = Flags::B;
        assert!(b > a && b >= a);
        assert!(a < b && a <= b);
    }

    fn hash<T: Hash>(t: &T) -> u64 {
        let mut s = DefaultHasher::new();
        t.hash(&mut s);
        s.finish()
    }

    #[test]
    fn test_hash() {
        let mut x = Flags::empty();
        let mut y = Flags::empty();
        assert_eq!(hash(&x), hash(&y));
        x = Flags::all();
        y = Flags::ABC;
        assert_eq!(hash(&x), hash(&y));
    }

    #[test]
    fn test_default() {
        assert_eq!(Flags::empty(), Flags::default());
    }

    #[test]
    fn test_debug() {
        assert_eq!(format!("{:?}", Flags::A | Flags::B), "Flags(A | B)");
        assert_eq!(format!("{:?}", Flags::empty()), "Flags(0x0)");
        assert_eq!(format!("{:?}", Flags::ABC), "Flags(A | B | C)");

        let extra = Flags::from_bits_retain(0xb8);

        assert_eq!(format!("{:?}", extra), "Flags(0xb8)");
        assert_eq!(format!("{:?}", Flags::A | extra), "Flags(A | 0xb8)");

        assert_eq!(
            format!("{:?}", Flags::ABC | extra),
            "Flags(A | B | C | ABC | 0xb8)"
        );

        assert_eq!(format!("{:?}", EmptyFlags::empty()), "EmptyFlags(0x0)");
    }

    #[test]
    fn test_display_from_str_roundtrip() {
        fn format_parse_case<T: fmt::Debug + fmt::Display + str::FromStr + PartialEq>(flags: T) where <T as str::FromStr>::Err: fmt::Display {
            assert_eq!(flags, {
                match flags.to_string().parse::<T>() {
                    Ok(flags) => flags,
                    Err(e) => panic!("failed to parse `{}`: {}", flags, e),
                }
            });
        }

        fn parse_case<T: fmt::Debug + str::FromStr + PartialEq>(expected: T, flags: &str) where <T as str::FromStr>::Err: fmt::Display + fmt::Debug {
            assert_eq!(expected, flags.parse::<T>().unwrap());
        }

        bitflags! {
            #[derive(Debug, Eq, PartialEq)]
            pub struct MultiBitFmtFlags: u8 {
                const A = 0b0000_0001u8;
                const B = 0b0001_1110u8;
            }
        }

        impl fmt::Display for MultiBitFmtFlags {
            fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
                fmt::Display::fmt(&self.0, f)
            }
        }

        impl str::FromStr for MultiBitFmtFlags {
            type Err = crate::parser::ParseError;

            fn from_str(s: &str) -> Result<Self, Self::Err> {
                Ok(MultiBitFmtFlags(s.parse()?))
            }
        }

        format_parse_case(FmtFlags::empty());
        format_parse_case(FmtFlags::all());
        format_parse_case(FmtFlags::고양이);
        format_parse_case(FmtFlags::고양이 | FmtFlags::개);
        format_parse_case(FmtFlags::물고기_고양이);
        format_parse_case(FmtFlags::from_bits_retain(0xb8));
        format_parse_case(FmtFlags::from_bits_retain(0x20));
        format_parse_case(MultiBitFmtFlags::from_bits_retain(3));

        parse_case(FmtFlags::empty(), "");
        parse_case(FmtFlags::empty(), " \r\n\t");
        parse_case(FmtFlags::empty(), "0x0");
        parse_case(FmtFlags::empty(), "0x0");

        parse_case(FmtFlags::고양이, "고양이");
        parse_case(FmtFlags::고양이, "  고양이  ");
        parse_case(FmtFlags::고양이, "고양이 | 고양이 | 고양이");
        parse_case(FmtFlags::고양이, "0x01");

        parse_case(FmtFlags::고양이 | FmtFlags::개, "고양이 | 개");
        parse_case(FmtFlags::고양이 | FmtFlags::개, "고양이|개");
        parse_case(FmtFlags::고양이 | FmtFlags::개, "\n고양이|개 ");

        parse_case(FmtFlags::고양이 | FmtFlags::물고기, "물고기_고양이");
    }

    #[test]
    fn test_from_str_err() {
        fn parse_case(pat: &str, flags: &str) {
            let err = flags.parse::<FmtFlags>().unwrap_err().to_string();
            assert!(err.contains(pat), "`{}` not found in error `{}`", pat, err);
        }

        parse_case("empty flag", "|");
        parse_case("empty flag", "|||");
        parse_case("empty flag", "고양이 |");
        parse_case("unrecognized named flag", "NOT_A_FLAG");
        parse_case("unrecognized named flag", "고양이 개");
        parse_case("unrecognized named flag", "고양이 | NOT_A_FLAG");
        parse_case("invalid hex flag", "0xhi");
        parse_case("invalid hex flag", "고양이 | 0xhi");
    }

    #[test]
    fn test_binary() {
        assert_eq!(format!("{:b}", Flags::ABC), "111");
        assert_eq!(format!("{:#b}", Flags::ABC), "0b111");
        let extra = Flags::from_bits_retain(0b1010000);
        assert_eq!(format!("{:b}", Flags::ABC | extra), "1010111");
        assert_eq!(format!("{:#b}", Flags::ABC | extra), "0b1010111");
    }

    #[test]
    fn test_octal() {
        assert_eq!(format!("{:o}", LongFlags::LONG_A), "177777");
        assert_eq!(format!("{:#o}", LongFlags::LONG_A), "0o177777");
        let extra = LongFlags::from_bits_retain(0o5000000);
        assert_eq!(format!("{:o}", LongFlags::LONG_A | extra), "5177777");
        assert_eq!(format!("{:#o}", LongFlags::LONG_A | extra), "0o5177777");
    }

    #[test]
    fn test_lowerhex() {
        assert_eq!(format!("{:x}", LongFlags::LONG_A), "ffff");
        assert_eq!(format!("{:#x}", LongFlags::LONG_A), "0xffff");
        let extra = LongFlags::from_bits_retain(0xe00000);
        assert_eq!(format!("{:x}", LongFlags::LONG_A | extra), "e0ffff");
        assert_eq!(format!("{:#x}", LongFlags::LONG_A | extra), "0xe0ffff");
    }

    #[test]
    fn test_upperhex() {
        assert_eq!(format!("{:X}", LongFlags::LONG_A), "FFFF");
        assert_eq!(format!("{:#X}", LongFlags::LONG_A), "0xFFFF");
        let extra = LongFlags::from_bits_retain(0xe00000);
        assert_eq!(format!("{:X}", LongFlags::LONG_A | extra), "E0FFFF");
        assert_eq!(format!("{:#X}", LongFlags::LONG_A | extra), "0xE0FFFF");
    }

    mod submodule {
        bitflags! {
            #[derive(Clone, Copy)]
            pub struct PublicFlags: i8 {
                const X = 0;
            }

            #[derive(Clone, Copy)]
            struct PrivateFlags: i8 {
                const Y = 0;
            }
        }

        #[test]
        fn test_private() {
            let _ = PrivateFlags::Y;
        }
    }

    #[test]
    fn test_public() {
        let _ = submodule::PublicFlags::X;
    }

    mod t1 {
        mod foo {
            pub type Bar = i32;
        }

        bitflags! {
            /// baz
            #[derive(Clone, Copy)]
            struct Flags: foo::Bar {
                const A = 0b00000001;
                #[cfg(foo)]
                const B = 0b00000010;
                #[cfg(foo)]
                const C = 0b00000010;
            }
        }
    }

    #[test]
    fn test_in_function() {
        bitflags! {
            #[derive(Clone, Copy, Debug, PartialEq, Eq)]
            struct Flags: u8 {
                const A = 1;
                #[cfg(any())] // false
                const B = 2;
            }
        }
        assert_eq!(Flags::all(), Flags::A);
        assert_eq!(format!("{:?}", Flags::A), "Flags(A)");
    }

    #[test]
    fn test_deprecated() {
        bitflags! {
            #[derive(Clone, Copy)]
            pub struct TestFlags: u32 {
                #[deprecated(note = "Use something else.")]
                const ONE = 1;
            }
        }
    }

    #[test]
    fn test_pub_crate() {
        mod module {
            bitflags! {
                #[derive(Clone, Copy)]
                pub (crate) struct Test: u8 {
                    const FOO = 1;
                }
            }
        }

        assert_eq!(module::Test::FOO.bits(), 1);
    }

    #[test]
    fn test_pub_in_module() {
        mod module {
            mod submodule {
                bitflags! {
                    // `pub (in super)` means only the module `module` will
                    // be able to access this.
                    #[derive(Clone, Copy)]
                    pub (in super) struct Test: u8 {
                        const FOO = 1;
                    }
                }
            }

            mod test {
                // Note: due to `pub (in super)`,
                // this cannot be accessed directly by the testing code.
                pub(super) fn value() -> u8 {
                    super::submodule::Test::FOO.bits()
                }
            }

            pub fn value() -> u8 {
                test::value()
            }
        }

        assert_eq!(module::value(), 1)
    }

    #[test]
    fn test_zero_value_flags() {
        bitflags! {
            #[derive(Clone, Copy, Debug, PartialEq, Eq)]
            struct Flags: u32 {
                const NONE = 0b0;
                const SOME = 0b1;
            }
        }

        assert!(Flags::empty().contains(Flags::NONE));
        assert!(Flags::SOME.contains(Flags::NONE));
        assert!(Flags::NONE.is_empty());

        assert_eq!(format!("{:?}", Flags::SOME), "Flags(NONE | SOME)");
    }

    #[test]
    fn test_empty_bitflags() {
        bitflags! {}
    }

    #[test]
    fn test_u128_bitflags() {
        bitflags! {
            #[derive(Clone, Copy, Debug, PartialEq, Eq)]
            struct Flags: u128 {
                const A = 0x0000_0000_0000_0000_0000_0000_0000_0001;
                const B = 0x0000_0000_0000_1000_0000_0000_0000_0000;
                const C = 0x8000_0000_0000_0000_0000_0000_0000_0000;
                const ABC = Self::A.bits() | Self::B.bits() | Self::C.bits();
            }
        }

        assert_eq!(Flags::ABC, Flags::A | Flags::B | Flags::C);
        assert_eq!(Flags::A.bits(), 0x0000_0000_0000_0000_0000_0000_0000_0001);
        assert_eq!(Flags::B.bits(), 0x0000_0000_0000_1000_0000_0000_0000_0000);
        assert_eq!(Flags::C.bits(), 0x8000_0000_0000_0000_0000_0000_0000_0000);
        assert_eq!(Flags::ABC.bits(), 0x8000_0000_0000_1000_0000_0000_0000_0001);
        assert_eq!(format!("{:?}", Flags::A), "Flags(A)");
        assert_eq!(format!("{:?}", Flags::B), "Flags(B)");
        assert_eq!(format!("{:?}", Flags::C), "Flags(C)");
        assert_eq!(format!("{:?}", Flags::ABC), "Flags(A | B | C)");
    }

    #[test]
    fn test_from_bits_edge_cases() {
        bitflags! {
            #[derive(Clone, Copy, Debug, PartialEq, Eq)]
            struct Flags: u8 {
                const A = 0b00000001;
                const BC = 0b00000110;
            }
        }

        let flags = Flags::from_bits(0b00000100);
        assert_eq!(flags, None);
        let flags = Flags::from_bits(0b00000101);
        assert_eq!(flags, None);
    }

    #[test]
    fn test_from_bits_truncate_edge_cases() {
        bitflags! {
            #[derive(Clone, Copy, Debug, PartialEq, Eq)]
            struct Flags: u8 {
                const A = 0b00000001;
                const BC = 0b00000110;
            }
        }

        let flags = Flags::from_bits_truncate(0b00000100);
        assert_eq!(flags, Flags::empty());
        let flags = Flags::from_bits_truncate(0b00000101);
        assert_eq!(flags, Flags::A);
    }

    #[test]
    fn test_iter() {
        bitflags! {
            #[derive(Clone, Copy, Debug, PartialEq, Eq)]
            struct Flags: u32 {
                const ONE  = 0b001;
                const TWO  = 0b010;
                const THREE = 0b100;
                #[cfg(windows)]
                const FOUR_WIN = 0b1000;
                #[cfg(unix)]
                const FOUR_UNIX = 0b10000;
                const FIVE = 0b01000100;
            }
        }

        let count = {
            #[cfg(any(unix, windows))]
            {
                5
            }

            #[cfg(not(any(unix, windows)))]
            {
                4
            }
        };

        let flags = Flags::all();
        assert_eq!(flags.into_iter().count(), count);

        for flag in flags.into_iter() {
            assert!(flags.contains(flag));
        }

        let mut iter = flags.iter_names();

        assert_eq!(iter.next().unwrap(), ("ONE", Flags::ONE));
        assert_eq!(iter.next().unwrap(), ("TWO", Flags::TWO));
        assert_eq!(iter.next().unwrap(), ("THREE", Flags::THREE));

        #[cfg(unix)]
        {
            assert_eq!(iter.next().unwrap(), ("FOUR_UNIX", Flags::FOUR_UNIX));
        }
        #[cfg(windows)]
        {
            assert_eq!(iter.next().unwrap(), ("FOUR_WIN", Flags::FOUR_WIN));
        }

        assert_eq!(iter.next().unwrap(), ("FIVE", Flags::FIVE));

        assert_eq!(iter.next(), None);

        let flags = Flags::empty();
        assert_eq!(flags.into_iter().count(), 0);

        let flags = Flags::ONE | Flags::THREE;
        assert_eq!(flags.into_iter().count(), 2);

        let mut iter = flags.iter_names();

        assert_eq!(iter.next().unwrap(), ("ONE", Flags::ONE));
        assert_eq!(iter.next().unwrap(), ("THREE", Flags::THREE));
        assert_eq!(iter.next(), None);

        let flags = Flags::from_bits_retain(0b1000_0000);
        assert_eq!(flags.into_iter().count(), 1);
        assert_eq!(flags.iter_names().count(), 0);
    }

    #[test]
    fn into_iter_from_iter_roundtrip() {
        let flags = Flags::ABC | Flags::from_bits_retain(0b1000_0000);

        assert_eq!(flags, flags.into_iter().collect::<Flags>());
    }

    #[test]
    fn test_from_name() {
        let flags = Flags::all();

        let mut rebuilt = Flags::empty();

        for (name, value) in flags.iter_names() {
            assert_eq!(value, Flags::from_name(name).unwrap());

            rebuilt |= Flags::from_name(name).unwrap();
        }

        assert_eq!(flags, rebuilt);
    }

    #[test]
    fn bits_types() {
        bitflags! {
            pub struct I8: i8 {
                const A = 1;
            }

            pub struct I16: i16 {
                const A = 1;
            }

            pub struct I32: i32 {
                const A = 1;
            }

            pub struct I64: i64 {
                const A = 1;
            }

            pub struct I128: i128 {
                const A = 1;
            }

            pub struct Isize: isize {
                const A = 1;
            }

            pub struct U8: u8 {
                const A = 1;
            }

            pub struct U16: u16 {
                const A = 1;
            }

            pub struct U32: u32 {
                const A = 1;
            }

            pub struct U64: u64 {
                const A = 1;
            }

            pub struct U128: u128 {
                const A = 1;
            }

            pub struct Usize: usize {
                const A = 1;
            }
        }
    }
}
