/* rglob.c
 *
 * Copyright (c) 1996-2005 Mike Gleason, NcFTP Software.
 * All rights reserved.
 *
 */

#include "syshdrs.h"
#ifdef PRAGMA_HDRSTOP
#	pragma hdrstop
#endif

#if (defined(WIN32) || defined(_WINDOWS)) && !defined(__CYGWIN__)
#define _CRT_SECURE_NO_WARNINGS 1
#endif


#ifndef HAVE_FNMATCH
static void
StripUnneccesaryGlobEntries(const FTPCIPtr cip, FTPLineListPtr fileList)
{
	FTPLinePtr lp, nextLine;
	const char *cp;

	for (lp=fileList->first; lp != NULL; lp = nextLine) {
		nextLine = lp->next;
		cp = strrchr(lp->line, '/');
		if (cp == NULL)
			cp = strrchr(lp->line, '\\');
		if (cp == NULL)
			cp = lp->line;
		else
			cp++;
		if ((strcmp(cp, ".") == 0) || (strcmp(cp, "..") == 0)) {
			PrintF(cip, "  Rglob omitted: [%s] (type 1)\n", lp->line);
			nextLine = RemoveLine(fileList, lp);
		}
	}
}	/* StripUnneccesaryGlobEntries */
#endif	/* !HAVE_FNMATCH */



int
PathContainsIntermediateDotDotSubDir(const char *s)
{
	int c;
	const char *a;

	if ((s[0] == '.') && (s[1] == '.')) {
		a = s + 2;
		while (*a == '.') a++;
		if (((*a == '/') || (*a == '\\')) || (*a == '\0'))
			return (1);
	}

	while (*s != '\0') {
		c = (int) *s++;
		if (((c == '/') || (c == '\\')) && (s[0] == '.') && (s[1] == '.')) {
			a = s + 2;

			/* Windows also treats '...', '....', etc as '..'
			 * so check for these too.  It doesn't matter on
			 * UNIX, but if those come back someone is up to
			 * no good.
			 */
			while (*a == '.') a++;
			if (((*a == '/') || (*a == '\\')) || (*a == '\0'))
				return (1);
		}
	}

	return (0);
}	/* PathContainsIntermediateDotDotSubDir */




/* We need to use this because using NLST gives us more stuff than
 * we want back sometimes.  For example, say we have:
 *
 * /a		(directory)
 * /a/b		(directory)
 * /a/b/b1
 * /a/b/b2
 * /a/b/b3
 * /a/c		(directory)
 * /a/c/c1
 * /a/c/c2
 * /a/c/c3
 * /a/file
 *
 * If you did an "echo /a/<star>" in a normal unix shell, you would expect
 * to get back /a/b /a/c /a/file.  But NLST gives back:
 *
 * /a/b/b1
 * /a/b/b2
 * /a/b/b3
 * /a/c/c1
 * /a/c/c2
 * /a/c/c3
 * /a/file
 *
 * So we use the following routine to convert into the format I expect.
 */

void
RemoteGlobCollapse(const FTPCIPtr cip, const char *pattern, FTPLineListPtr fileList)
{
	FTPLinePtr lp, nextLine;
	char *patPrefix;
	char *patDir;
	char *cur, *prev;
	char *cp;
	char *newpath;
	size_t plen;

	/* Copy all characters before and including the last path delimiter. */
	patDir = NULL;	
	cp = StrRFindLocalPathDelim(pattern);
	if (cp != NULL) {
		patDir = StrDup(pattern);
		if (patDir == NULL)
			return;
		patDir[(cp - pattern) + 1] = '\0';
	}

	/* Copy all characters before the first glob-char. */
	cp = strpbrk(pattern, kGlobChars);
	patPrefix = StrDup(pattern);
	if (patPrefix == NULL) {
		free(patDir);
		return;
	}
	if (cp != NULL) {
		plen = (size_t) (cp - pattern);
		patPrefix[plen] = '\0';
	} else {
		plen = strlen(patPrefix);
	}

	cur = prev = NULL;
	for (lp=fileList->first; lp != NULL; lp = nextLine) {
		nextLine = lp->next;
		if (ISTRNEQ(lp->line, patPrefix, plen)) {
			if (Dynsrecpy(&cur, lp->line + plen, 0) == NULL)
				goto done;
			cp = strpbrk(cur, "/\\");
			if (cp != NULL)
				*cp = '\0';
			if ((prev != NULL) && (STREQ(cur, prev))) {
				PrintF(cip, "  Rglob omitted: [%s] (type 2)\n", lp->line);
				nextLine = RemoveLine(fileList, lp);
			} else if (PathContainsIntermediateDotDotSubDir(lp->line + plen)) {
				PrintF(cip, "  Rglob omitted: [%s] (type 3)\n", lp->line);
				nextLine = RemoveLine(fileList, lp);
			} else {
				if (Dynsrecpy(&prev, cur, 0) == NULL)
					goto done;

#ifdef HAVE_SNPRINTF
				if (Dynsrecpy(&lp->line, patPrefix, cur, 0) == NULL)
					goto done;
#else
				/* We are playing with a dynamically
				 * allocated string, but since the
				 * following expression is guaranteed
				 * to be the same or shorter, we won't
				 * overwrite the bounds.
				 */
				(void) sprintf(lp->line, "%s%s", patPrefix, cur);
#endif
			}
		} else if (strpbrk(lp->line, "/\\") == NULL) {
			if (patDir != NULL) {
				newpath = NULL;
				if (Dynsrecpy(&newpath, patDir, lp->line, 0) == NULL)
					goto done;
				PrintF(cip, "  Rglob changed: [%s] to [%s]\n", lp->line, newpath);
				free(lp->line);
				lp->line = newpath;
			}
		} else {
			PrintF(cip, "  Rglob omitted: [%s] (type 4)\n", lp->line);
			nextLine = RemoveLine(fileList, lp);
		}
	}

done:
	StrFree(&patDir);
	StrFree(&patPrefix);
	StrFree(&cur);
	StrFree(&prev);
}	/* RemoteGlobCollapse */


#ifdef HAVE_FNMATCH
static void
RemoteGlobFilter(const FTPCIPtr cip, const char *pardir, const char *const dirsep, const char *const pattern, FTPLineListPtr fileList)
{
	FTPLinePtr lp = NULL, nextLine = NULL;
	const char *basename;
	const char *linestart;
	char *match = NULL;
	char *lineFixed = NULL;
	size_t pardirlen = 0;
	int havePardirInOutput = 0;
	int haveRootPardir = 0;

	if (pardir != NULL) {
		if (IsLocalPathDelim(*pardir)) {
			while (IsLocalPathDelim(*pardir)) { pardir++; }
			--pardir;
			pardirlen = strlen(pardir);	/* can be "" */
			if (pardirlen == 1) {
				haveRootPardir = 1;
				/* PrintF(cip, "Rglob: haveRootPardir\n"); */
			}
		} else {
			pardirlen = strlen(pardir);	/* can be "" */
		}
	}

	if ((pattern != NULL) && (pattern[0] != '\0') && (fileList != NULL) && (cip != NULL)) {
		for (lp=fileList->first; lp != NULL; lp = nextLine) {
			nextLine = lp->next;

			linestart = lp->line;
			if (IsLocalPathDelim(*linestart)) {
				while (IsLocalPathDelim(*linestart)) { linestart++; }
				--linestart;
				if (linestart > lp->line) {
					(void) Dynscpy(&lineFixed, linestart, NULL);
					(void) Dynsrecpy(&lp->line, lineFixed, NULL);
					StrFree(&lineFixed);
					linestart = lp->line;
				}
			}

			basename = NULL;
			if (pardirlen > 0) {
				if (strncasecmp(linestart, pardir, pardirlen) == 0) {
					basename = linestart + pardirlen + (haveRootPardir ? 0 : 1) /* dirsep */;
					havePardirInOutput = 1;
				} else {
					havePardirInOutput = 0;
				}
			}

			if (basename == NULL) {
				basename = linestart;
			}

			/*
			PrintF(cip, "Rglob: fnmatch pattern=\"%s\" basename=\"%s\" hvPdIO=%d hvRPd=%d\n", pattern, basename, havePardirInOutput, haveRootPardir);
			*/
			if (fnmatch(pattern, basename, 	(
								0
#ifdef FNM_PATHNAME
								| FNM_PATHNAME
#endif
#ifdef FNM_EXTMATCH
								| FNM_EXTMATCH
#endif
#ifdef FNM_CASEFOLD
								| FNM_CASEFOLD
#endif
							)
			) != 0) {
				/* No match, or error. */
				PrintF(cip, "  - Rglob [%s]\n", lp->line);
				nextLine = RemoveLine(fileList, lp);
			} else {
				if ((pardirlen > 0) && (havePardirInOutput == 0)) {
					/* Prepend pardir to each match */
					/*
					PrintF(cip, "Rglob: prepend pardir=[%s] sep=[%s] basename=[%s]\n", pardir, (haveRootPardir ? "" : dirsep), basename);
					*/
					(void) Dynscpy(&match, pardir, (haveRootPardir ? "" : dirsep), basename, NULL);
					(void) Dynsrecpy(&lp->line, match, NULL);
					StrFree(&match);
				}
			}
		}
	}
}	/* RemoteGlobFilter */
#endif	/* HAVE_FNMATCH */




int
FTPRemoteGlob(FTPCIPtr cip, FTPLineListPtr fileList, const char *pattern, int doGlob)
{
	char *cp;
	const char *lsflags;
	FTPLinePtr lp;
	int result = kNoErr;
	char *patParentDir = NULL;
	char *patBase = NULL;
	char pathDelim[2];

	if (cip == NULL)
		return (kErrBadParameter);
	if (strcmp(cip->magic, kLibraryMagic))
		return (kErrBadMagic);

	if (fileList == NULL)
		return (kErrBadParameter);
	InitLineList(fileList);

	if ((pattern == NULL) || (pattern[0] == '\0'))
		return (kErrBadParameter);

	pathDelim[0] = '/';
	pathDelim[1] = '\0';

	if ((doGlob == 1) && (GLOBCHARSINSTR(pattern))) {
		/* Have glob chars, but we consider two different cases:
		 * (1) No directory separators present, e.g. "file*.txt"
		 *   NLST
		 *   filter the results for file*.txt:
		 *     YES: file1.txt -> file1.txt
		 *     YES: file2.txt -> file2.txt
		 *     NO:  test3.txt
		 * (2a) Directory separators present, "/dir1/dir2/file*.txt"
		 *   NLST /dir1/dir2
		 *   filter the results for file*.txt, prepend /dir1/dir2/ to each match:
		 *     YES: file1.txt -> /dir1/dir2/file1.txt
		 *     YES: file2.txt -> /dir1/dir2/file2.txt
		 *     NO:  test3.txt
		 * (2b) Directory separators present, "/dir1/dir2/file*.txt"
		 *   NLST /dir1/dir2
		 *   filter the results with dir prefix for file*.txt:
		 *     YES: /dir1/dir2/file1.txt -> /dir1/dir2/file1.txt
		 *     YES: /dir1/dir2/file2.txt -> /dir1/dir2/file2.txt
		 *     NO:  /dir1/dir2/test3.txt
		 */

		/* Use NLST, which lists files one per line. */
		lsflags = "";
		
		/* Optimize for "NLST *" case which is same as "NLST". */
		if (strcmp(pattern, "*") == 0) {
			pattern = "";
		} else if (strcmp(pattern, "**") == 0) {
			/* Hack; Lets you try "NLST -a" if you're daring. */
			/* Need to use "NLST -a" whenever possible,
			 * because wu-ftpd doesn't do NLST right, IMHO.
			 * (It doesn't include directories in the NLST
			 *  if you do "NLST /the/dir" without -a.)
			 */
			pattern = "";
			lsflags = (cip->hasNLST_a == kCommandNotAvailable) ? "" : "-a";
		}

		if (Dynscpy(&patParentDir, pattern, (char *) 0) == NULL)
			return (kErrBadParameter);
		cp = StrRFindLocalPathDelim(patParentDir);
		if (cp == patParentDir) {
			/* Had a root path like "/foobar*" */
			pathDelim[0] = '\0';
			cp++;
			if (Dynscpy(&patBase, cp, (char *) 0) == NULL) {
				result = kErrBadParameter;
				goto rgerr;
			}
			*cp = '\0';
		} else if (cp != NULL) {
			/* Have at least one subdirectory like "/foo/bar*" */
			pathDelim[0] = *cp;
			pathDelim[1] = '\0';
			*cp++ = '\0';
			if (Dynscpy(&patBase, cp, (char *) 0) == NULL) {
				result = kErrBadParameter;
				goto rgerr;
			}
		} else {
			*patParentDir = '\0';
			if (Dynscpy(&patBase, pattern, (char *) 0) == NULL) {
				result = kErrBadParameter;
				goto rgerr;
			}
		}
		/* PrintF(cip, "pattern=[%s] patParentDir=[%s] patBase=[%s]\n", pattern, patParentDir, patBase); */
		if ((result = FTPListToMemory2(cip, patParentDir, fileList, lsflags, 0, (int *) 0)) < 0) {
			if (*lsflags == '\0')
				goto rgerr;
			if (strchr(lsflags, 'a') != NULL) {
				/* Try again, without "-a" */
				cip->hasNLST_a = kCommandNotAvailable;
				lsflags = "";
				if ((result = FTPListToMemory2(cip, patParentDir, fileList, lsflags, 0, (int *) 0)) < 0) {
					goto rgerr;
				}
				/* else proceed */
			} else {
				goto rgerr;
			}
		}
		if (fileList->first == NULL) {
			cip->errNo = kErrGlobNoMatch;
			PrintF(cip, "  (Rglob no matches)\n");
			goto rgerr;
		}
		if (fileList->first == fileList->last) {
#define glberr(a) (ISTRNEQ(cp, a, strlen(a)))
			/* If we have only one item in the list, see if it really was
			 * an error message we would recognize.
			 */
			cp = strchr(fileList->first->line, ':');
			if (cp != NULL) {
				if (glberr(": No such file or directory")) {
					(void) RemoveLine(fileList, fileList->first);
					result = kErrGlobFailed;
					goto rgerr;
				} else if (glberr(": No match")) {
					result = kErrGlobNoMatch;
					goto rgerr;
				}
			}
		}
#ifdef HAVE_FNMATCH
		RemoteGlobFilter(cip, patParentDir, pathDelim, patBase, fileList);
#else
		StripUnneccesaryGlobEntries(cip, fileList);
		RemoteGlobCollapse(cip, pattern, fileList);
#endif	/* HAVE_FNMATCH */

		if (fileList->first == NULL) {
			result = kErrGlobNoMatch;
			PrintF(cip, "  (Rglob no matches)\n");
			goto rgerr;
		}

		for (lp=fileList->first; lp != NULL; lp = lp->next)
			PrintF(cip, "  + Rglob [%s]\n", lp->line);
		result = kNoErr;
	} else {
		/* Or, if there were no globbing characters in 'pattern', then the
		 * pattern is really just a filename.  So for this case the
		 * file list is really just a single file.
		 */
		fileList->first = fileList->last = NULL;
		(void) AddLine(fileList, pattern);
	}

rgerr:
	StrFree(&patParentDir);
	StrFree(&patBase);
	if (result != kNoErr) cip->errNo = result;
	return (result);
}	/* FTPRemoteGlob */
