/*
 * Check: a unit test framework for C
 * Copyright (C) 2020 Wander Lairson Costa
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include "libcompat.h"

#ifdef HAVE_WIN32_INIT_ONCE

static int mutex_init(pthread_mutex_t *mutex)
{
    BOOL pending;
    int ret = 0;

    if (!InitOnceBeginInitialize(&mutex->init, 0, &pending, NULL))
        return -1;

    if (pending)
    {
        mutex->mutex = CreateMutexW(NULL, FALSE, NULL);
        if (!mutex->mutex)
            ret = -1;
    }

    InitOnceComplete(&mutex->init, 0, NULL);

    return ret;
}

int pthread_mutex_init(pthread_mutex_t *mutex, pthread_mutexattr_t *attr)
{
    InitOnceInitialize(&mutex->init);
    return mutex_init(mutex);
}

int pthread_mutex_destroy(pthread_mutex_t *mutex)
{
    return CloseHandle(mutex->mutex) ? 0 : -1;
}

int pthread_mutex_lock(pthread_mutex_t *mutex)
{
    if (mutex_init(mutex) != 0)
        return -1;

    return WaitForSingleObject(mutex->mutex, INFINITE) != WAIT_OBJECT_0
        ? -1 : 0;
}

int pthread_mutex_unlock(pthread_mutex_t *mutex)
{
    return ReleaseMutex(mutex->mutex) ? 0 : -1;
}

#endif
